"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.vendorService = exports.VendorService = void 0;
const supabase_1 = require("@/config/supabase");
/**
 * Vendor Service - Handles vendor-related database operations
 */
class VendorService {
    /**
     * Create a new vendor record
     */
    async createVendor(vendorData) {
        try {
            const { data, error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .insert([vendorData])
                .select()
                .single();
            return { data, error };
        }
        catch (error) {
            console.error('Create vendor error:', error);
            return { data: null, error };
        }
    }
    /**
     * Get vendor by user ID
     */
    async getVendorByUserId(userId) {
        try {
            const { data, error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .select('*')
                .eq('user_id', userId)
                .single();
            return { data, error };
        }
        catch (error) {
            console.error('Get vendor by user ID error:', error);
            return { data: null, error };
        }
    }
    /**
     * Get vendor by ID
     */
    async getVendorById(vendorId) {
        try {
            const { data, error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .select('*')
                .eq('id', vendorId)
                .single();
            return { data, error };
        }
        catch (error) {
            console.error('Get vendor by ID error:', error);
            return { data: null, error };
        }
    }
    /**
     * Update vendor onboarding information
     */
    async updateOnboarding(vendorId, onboardingData) {
        try {
            const { data, error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .update({
                business_name: onboardingData.business_name,
                profile_picture_url: onboardingData.profile_picture_url || null,
                onboarding_status: supabase_1.ONBOARDING_STATUS.COMPLETE,
                updated_at: new Date().toISOString()
            })
                .eq('id', vendorId)
                .select()
                .single();
            return { data, error };
        }
        catch (error) {
            console.error('Update onboarding error:', error);
            return { data: null, error };
        }
    }
    /**
     * Update vendor profile
     */
    async updateVendor(vendorId, updateData) {
        try {
            const { data, error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .update({
                ...updateData,
                updated_at: new Date().toISOString()
            })
                .eq('id', vendorId)
                .select()
                .single();
            return { data, error };
        }
        catch (error) {
            console.error('Update vendor error:', error);
            return { data: null, error };
        }
    }
    /**
     * Get all vendors (admin only)
     */
    async getAllVendors() {
        try {
            const { data, error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .select('*')
                .order('created_at', { ascending: false });
            return { data, error };
        }
        catch (error) {
            console.error('Get all vendors error:', error);
            return { data: null, error };
        }
    }
    /**
     * Delete vendor (admin only)
     */
    async deleteVendor(vendorId) {
        try {
            const { error } = await supabase_1.supabase
                .from(supabase_1.TABLES.VENDORS)
                .delete()
                .eq('id', vendorId);
            return { error };
        }
        catch (error) {
            console.error('Delete vendor error:', error);
            return { error };
        }
    }
}
exports.VendorService = VendorService;
exports.vendorService = new VendorService();
//# sourceMappingURL=service.js.map